#########################################################################
# test_task_hanningsmooth.py
# Copyright (C) 2018
# Associated Universities, Inc. Washington DC, USA.
#
# This script is free software; you can redistribute it and/or modify it
# under the terms of the GNU Library General Public License as published by
# the Free Software Foundation; either version 2 of the License, or (at your
# option) any later version.
#
# This library is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
# FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
# License for more details.
#
#
# Based on the requirements listed in casadocs found here:
# https://casadocs.readthedocs.io/en/stable/api/tt/casatasks.manipulation.hanningsmooth.html
#
##########################################################################
import os
import shutil
import unittest

from casatasks.private.parallel.parallel_data_helper import ParallelDataHelper
from casatasks import hanningsmooth, mstransform, partition, cvel, split, clearcal
from casatools import ctsys, ms
from casatestutils import testhelper as th

# Path for data
datapath = ctsys.resolve('unittest/hanningsmooth/')

'''
functional tests for task hanningsmooth
'''

# Pick up alternative data directory to run tests on MMSs
testmms = False
if 'TEST_DATADIR' in os.environ:   
    DATADIR = str(os.environ.get('TEST_DATADIR'))+'/hanningsmooth/'
    if os.path.isdir(DATADIR):
        testmms = True
        datapath = DATADIR

print('hanningsmooth tests will use data from %s' % datapath)

class test_base(unittest.TestCase):

    def setUp_ngc5921(self):
        # only DATA column
        self.msfile = 'ngc5921_ut.ms'
        if testmms:
            self.msfile = 'ngc5921_ut.mms'
            
        if (not os.path.exists(self.msfile)):
            shutil.copytree(os.path.join(datapath,self.msfile), self.msfile)

    def setUp_almams(self):
        # MS with DATA and CORRECTED_DATA
        self.msfile = 'ALMA-data-mst-science-testing-CAS-5013-one-baseline-one-timestamp.ms'
        if testmms:
            self.msfile = 'ALMA-data-mst-science-testing-CAS-5013-one-baseline-one-timestamp.mms'
            
        if (not os.path.exists(self.msfile)):
            shutil.copytree(os.path.join(datapath,self.msfile), self.msfile)

    def setUp_FourAnts(self):
        datadir = os.path.join('unittest', 'hanningsmooth')
        ms_Four_ants = 'Four_ants_3C286.ms'
        self.msfile = ms_Four_ants

        if (not os.path.exists(self.msfile)):
            datapath_simple = ctsys.resolve(os.path.join(datadir, ms_Four_ants))
            shutil.copytree(datapath_simple, ms_Four_ants)

    def createMMS(self, msfile, column='data', axis='auto',scans='',spws=''):
        '''Create MMSs for tests with input MMS'''
        prefix = msfile.rstrip('.ms')
        if not os.path.exists(msfile):
            os.system('cp -RL '+os.path.join(datapath,msfile)+' '+ msfile)
        
        # Create an MMS for the tests
        self.testmms = prefix + ".test.mms"

        if os.path.exists(self.testmms):
            os.system("rm -rf " + self.testmms)
            
        print("................. Creating test MMS ..................")
        partition(vis=msfile, outputvis=self.testmms, datacolumn=column,
                    createmms=True,separationaxis=axis, scan=scans, spw=spws)


class hanningsmooth_test1(test_base):
    
    def setUp(self):
        self.setUp_ngc5921()

    def tearDown(self):
        if (os.path.exists(self.outputms)):
            shutil.rmtree(self.outputms,ignore_errors=True)        
        
    def test1(self):
        """hanningsmooth - Test 1: Wrong input MS should raise an exception"""
        msfile = 'badmsfile'
        self.outputms = 'none.ms'
        passes = False
        # CASA5 tasks do not throw exceptions, they return a value of False
        try:
            ret = hanningsmooth(vis=msfile)
            if not ret:
                passes = True
        except:
            passes = True
        self.assertTrue(passes)
        
    def test2(self):
        '''hanningsmooth - Test 2: Check that output MS is created'''
        self.outputms = 'hann2.ms'
        hanningsmooth(vis=self.msfile, outputvis=self.outputms, datacolumn='corrected')
        # Smoothed data should be saved in DATA column of outupt MS
        self.assertTrue(os.path.exists(self.outputms))
                
    def test3(self):
        '''hanningsmooth - Test 3: Check theoretical and calculated values on non-existing CORRECTED column'''
        self.outputms = 'hann3.ms'

      # check correct flagging (just for one row as a sample)
        flag_col = th.getVarCol(self.msfile, 'FLAG')
        self.assertTrue(flag_col['r1'][0][0] == [False])
        self.assertTrue(flag_col['r1'][0][1] == [False])
        self.assertTrue(flag_col['r1'][0][61] == [False])
        self.assertTrue(flag_col['r1'][0][62] == [False])

        # It should fall-back and use the input DATA column
        hanningsmooth(vis=self.msfile, outputvis=self.outputms, datacolumn='corrected')

      # check correct flagging (just for one row as a sample)
        flag_col = th.getVarCol(self.outputms, 'FLAG')
        self.assertTrue(flag_col['r1'][0][0] == [True])
        self.assertTrue(flag_col['r1'][0][1] == [False])
        self.assertTrue(flag_col['r1'][0][61] == [False])
        self.assertTrue(flag_col['r1'][0][62] == [True])

        data_col = th.getVarCol(self.msfile, 'DATA')
        corr_col = th.getVarCol(self.outputms, 'DATA')
        nrows = len(corr_col)
        
      # Loop over every 2nd row,pol and get the data for each channel
        max = 1e-05
        for i in range(1,nrows,2) :
            row = 'r%s'%i            
            # polarization is 0-1
            for pol in range(0,2) :
                # array's channels is 0-63
                for chan in range(1,62) :
                    # channels must start from second and end before the last
                    data = data_col[row][pol][chan]
                    dataB = data_col[row][pol][chan-1]
                    dataA = data_col[row][pol][chan+1]
        
                    Smoothed = th.calculateHanning(dataB,data,dataA)
                    CorData = corr_col[row][pol][chan]
                    
                    # Check the difference
                    self.assertTrue(abs(CorData-Smoothed) < max )

    def test4(self):
        '''hanningsmooth - Test 4: Theoretical and calculated values should be the same for MMS-case'''
    
        # Split the input to decrease the running time
        split(self.msfile, outputvis='splithan.ms',scan='1,2',datacolumn='data')
        self.msfile = 'splithan.ms'
        
        # create a test MMS. It creates self.testmms
        self.createMMS(self.msfile)
        self.outputms = 'hann4.mms'
        
      # check correct flagging (just for one row as a sample)
        mslocal = ms()
        mslocal.open(self.msfile)
        mslocal.sort('sorted.ms',['OBSERVATION_ID','ARRAY_ID','SCAN_NUMBER','FIELD_ID','DATA_DESC_ID','ANTENNA1','ANTENNA2','TIME'])
        mslocal.close()
        self.msfile = 'sorted.ms'
        flag_col = th.getVarCol(self.msfile, 'FLAG')
        self.assertTrue(flag_col['r1'][0][0] == [False])
        self.assertTrue(flag_col['r1'][0][1] == [False])
        self.assertTrue(flag_col['r1'][0][61] == [False])
        self.assertTrue(flag_col['r1'][0][62] == [False])
        
        data_col = th.getVarCol(self.msfile, 'DATA')        
        hanningsmooth(vis=self.testmms, outputvis=self.outputms, datacolumn='data', keepmms=True)
        self.assertTrue(ParallelDataHelper.isParallelMS(self.outputms), 'Output should be an MMS')

      # Sort the MMS
        mslocal.open(self.outputms)
        mslocal.sort('sorted.mms',['OBSERVATION_ID','ARRAY_ID','SCAN_NUMBER','FIELD_ID','DATA_DESC_ID','ANTENNA1','ANTENNA2','TIME'])
        mslocal.close()
        self.outputms = 'sorted.mms'
        
        corr_col = th.getVarCol(self.outputms, 'DATA')
        nrows = len(corr_col)

      # check correct flagging (just for one row as a sample)
        flag_col = th.getVarCol(self.outputms, 'FLAG')
        self.assertTrue(flag_col['r1'][0][0] == [True])
        self.assertTrue(flag_col['r1'][0][1] == [False])
        self.assertTrue(flag_col['r1'][0][61] == [False])
        self.assertTrue(flag_col['r1'][0][62] == [True])
        
      # Loop over every 2nd row,pol and get the data for each channel
        max = 1e-05
        for i in range(1,nrows,2) :
            row = 'r%s'%i            
            # polarization is 0-1
            for pol in range(0,2) :
                # array's channels is 0-63
                for chan in range(1,62) :
                    # channels must start from second and end before the last
                    data = data_col[row][pol][chan]
                    dataB = data_col[row][pol][chan-1]
                    dataA = data_col[row][pol][chan+1]
        
                    Smoothed = th.calculateHanning(dataB,data,dataA)
                    CorData = corr_col[row][pol][chan]
                    
                    # Check the difference
                    self.assertTrue(abs(CorData-Smoothed) < max )

    def test6(self):
        '''hanningsmooth - Test 6: Flagging should be correct with datacolumn==ALL'''
        self.outputms = 'hann6.ms'
        
      # check correct flagging (just for one row as a sample)
        flag_col = th.getVarCol(self.msfile, 'FLAG')
        self.assertTrue(flag_col['r1'][0][0] == [False])
        self.assertTrue(flag_col['r1'][0][1] == [False])
        self.assertTrue(flag_col['r1'][0][61] == [False])
        self.assertTrue(flag_col['r1'][0][62] == [False])

        hanningsmooth(vis=self.msfile,outputvis=self.outputms, datacolumn='all')

      # check correct flagging (just for one row as a sample)
        flag_col = th.getVarCol(self.outputms, 'FLAG')
        self.assertTrue(flag_col['r1'][0][0] == [True])
        self.assertTrue(flag_col['r1'][0][1] == [False])
        self.assertTrue(flag_col['r1'][0][61] == [False])
        self.assertTrue(flag_col['r1'][0][62] == [True])

    def test7(self):
        '''hanningsmooth - Test 7: Flagging should be correct when hanning smoothing within cvel (no transform)'''
        self.outputms = 'cvelngc.ms'
        clearcal(vis=self.msfile)
        
      # check correct flagging (just for one row as a sample)
        flag_col = th.getVarCol(self.msfile, 'FLAG')
        self.assertTrue(flag_col['r1'][0][0] == [False])
        self.assertTrue(flag_col['r1'][0][1] == [False])
        self.assertTrue(flag_col['r1'][0][61] == [False])
        self.assertTrue(flag_col['r1'][0][62] == [False])

        cvel(vis=self.msfile, outputvis=self.outputms, hanning=True)

      # check correct flagging (just for one row as a sample)
        flag_col = th.getVarCol(self.outputms, 'FLAG')
        self.assertTrue(flag_col['r1'][0][0] == [True])
        self.assertTrue(flag_col['r1'][0][1] == [False])
        self.assertTrue(flag_col['r1'][0][61] == [False])
        self.assertTrue(flag_col['r1'][0][62] == [True])

    def test8(self):
        '''hanningsmooth - Test 8: Flagging should be correct when hanning smoothing within mstransform (with regrid)'''
        self.outputms = 'cvelngc.ms'
        
      # check correct flagging (just for one row as a sample)
        flag_col = th.getVarCol(self.msfile, 'FLAG')
        self.assertTrue(flag_col['r1'][0][0] == [False])
        self.assertTrue(flag_col['r1'][0][1] == [False])
        self.assertTrue(flag_col['r1'][0][61] == [False])
        self.assertTrue(flag_col['r1'][0][62] == [False])

        # CAS-4114 cvel doesn't support MMS. Compare with mstransform instead.
#        cvel(vis=self.msfile, outputvis=self.outputms, hanning=True, outframe='cmb')
        mstransform(vis=self.msfile, outputvis=self.outputms, datacolumn='data',
                    hanning=True, regridms=True, outframe='cmb')

        # check correct flagging (just for one row as a sample)
        flag_col = th.getVarCol(self.outputms, 'FLAG')
        self.assertTrue(flag_col['r1'][0][0] == [True])
        self.assertTrue(flag_col['r1'][0][1] == [False])
        self.assertTrue(flag_col['r1'][0][2] == [False])
        self.assertTrue(flag_col['r1'][0][60] == [False])
        self.assertTrue(flag_col['r1'][0][61] == [True])
        self.assertTrue(flag_col['r1'][0][62] == [True])


class hanningsmooth_test2(test_base):
    
    def setUp(self):
        self.setUp_almams()

    def tearDown(self):
        if (os.path.exists(self.outputms)):
            shutil.rmtree(self.outputms,ignore_errors=True)       
             
    def test_default_cols(self):
        '''hanningsmooth: Default datacolumn=all and MMS output'''
        
        self.createMMS(self.msfile,column='all')
        self.outputms = 'hannall.ms'

        hanningsmooth(vis=self.testmms, outputvis=self.outputms)
        self.assertTrue(ParallelDataHelper.isParallelMS(self.outputms), 'Output should be an MMS')
        
        # Should have all scratch columns in output
        cd = th.getColDesc(self.outputms, 'DATA')
        self.assertGreater(len(cd), 0, 'DATA column does not exist')
        cc = th.getColDesc(self.outputms, 'CORRECTED_DATA')
        self.assertGreater(len(cc), 0, 'CORRECTED_DATA does not exist')
        
        # Now repeat the above steps but create an output MS by setting keepmms=False
        os.system('rm -rf '+self.outputms)
        hanningsmooth(vis=self.testmms, outputvis=self.outputms, keepmms=False)
        self.assertFalse(ParallelDataHelper.isParallelMS(self.outputms), 'Output should be a normal MS')
        
        # Should have all scratch columns in output
        cd = th.getColDesc(self.outputms, 'DATA')
        self.assertGreater(len(cd), 0, 'DATA column does not exist')
        cc = th.getColDesc(self.outputms, 'CORRECTED_DATA')
        self.assertGreater(len(cc), 0, 'CORRECTED_DATA does not exist')

    def test_corrected_col(self):
        '''hanningsmooth: Apply smoothing in CORRECTED column'''
        self.outputms = 'hanncorr.ms'

      # check correct flagging before (just for one row as a sample)
        flag_col = th.getVarCol(self.msfile, 'FLAG')
        self.assertTrue(flag_col['r1'][0][0] == [False])
        self.assertTrue(flag_col['r1'][0][1] == [False])
        self.assertTrue(flag_col['r1'][0][3838] == [False])
        self.assertTrue(flag_col['r1'][0][3839] == [False])
        
        # input column
        data_col = th.getVarCol(self.msfile, 'CORRECTED_DATA') 
               
        hanningsmooth(vis=self.msfile, outputvis=self.outputms, datacolumn='corrected')
        
        # output smoothed column
        corr_col = th.getVarCol(self.outputms, 'DATA')
        nrows = len(corr_col)
        
      # check correct flagging after (just for one row as a sample)
        flag_col = th.getVarCol(self.outputms, 'FLAG')
        self.assertTrue(flag_col['r1'][0][0] == [True])
        self.assertTrue(flag_col['r1'][0][1] == [False])
        self.assertTrue(flag_col['r1'][0][3838] == [False])
        self.assertTrue(flag_col['r1'][0][3839] == [True])

      # Loop over every 2nd row,pol and get the data for each channel
        max = 1e-04
        for i in range(1,nrows,2) :
            row = 'r%s'%i            
            # polarization is 0-1
            for pol in range(0,2) :
                # array's channels is 0-3840
                for chan in range(1,3839) :
                    # channels must start from second and end before the last
                    data = data_col[row][pol][chan]
                    dataB = data_col[row][pol][chan-1]
                    dataA = data_col[row][pol][chan+1]
        
                    Smoothed = th.calculateHanning(dataB,data,dataA)
                    CorData = corr_col[row][pol][chan]
                    
                    # Check the difference
                    self.assertTrue(abs(CorData-Smoothed) < max, 
                                    'CorData=%s Smoothed=%s in row=%s pol=%s chan=%s'%(CorData,Smoothed,row,pol,chan))

class hanningsmooth_test_skipspw(test_base):

    def setUp(self):
        self.setUp_FourAnts()

    def tearDown(self):
        if (os.path.exists(self.outputms)):
            shutil.rmtree(self.outputms,ignore_errors=True)

    def _compare_results(self, inpms, outms, smooth_spw, spw_sel=None):
        # Loop over every 2nd row,pol and get the data for each channel
        data_col = th.getVarCol(inpms, 'DATA')
        corr_col = th.getVarCol(outms, 'DATA')
        flag_col = th.getVarCol(self.msfile, 'FLAG')
        flag_out = th.getVarCol(self.outputms, 'FLAG')
        spwcolinp = th.getVarCol(inpms, 'DATA_DESC_ID')
        spwcol = th.getVarCol(outms, 'DATA_DESC_ID')
        nrows = len(corr_col)

        max = 1e-05
        inp_offset = 0
        for i in range(1,nrows,2) :
            row = 'r%s'%i
            rowinp = 'r%s'%(i+inp_offset)
            smoothed = False
            if spwcol[row] in smooth_spw:
                smoothed = True
            if spw_sel is not None :
                while spwcol[row] != spwcolinp[rowinp] :
                    inp_offset += 1
                    rowinp = 'r%s'%(i+inp_offset)
            # polarization is 0-1
            for pol in range(0,2) :
                nchan = len(data_col[row][pol])
                for chan in range(1,nchan-1) :
                    # channels must start from second and end before the last
                    data = data_col[rowinp][pol][chan]
                    dataB = data_col[rowinp][pol][chan-1]
                    dataA = data_col[rowinp][pol][chan+1]

                    Smoothed = th.calculateHanning(dataB,data,dataA)
                    CorData = corr_col[row][pol][chan]

                    # Check the difference
                    if not smoothed :
                        if  not (abs(CorData-data) < max ) :
                            self.assertTrue(abs(CorData-data) < max,
                                            'CorData=%s data=%s in row=%s pol=%s chan=%s'%(CorData,data,row,pol,chan))
                        if  not (flag_col[rowinp][pol][chan] == flag_out[row][pol][chan]) :
                            self.assertTrue(flag_col[rowinp][pol][chan] == flag_out[row][pol][chan],
                                            'flag=%s in row=%s pol=%s chan=%s'%(flag_col[rowinp][pol][chan],row,pol,chan))
                    else :
                        if  not (abs(CorData-Smoothed) < max ) :
                            self.assertTrue(abs(CorData-Smoothed) < max,
                                            'CorData=%s Smoothed=%s in row=%s pol=%s chan=%s'%(CorData,Smoothed,row,pol,chan))

    def test_smooth_spw_intvec(self):
        '''hanningsmooth: Use vector integer parameter smooth_spw to select only some of the SPWs to be smoothed'''

        self.outputms = 'hanning_smooth_spw.ms'
        smooth_spw = [6,7,8,9]
        hanningsmooth(vis=self.msfile, outputvis=self.outputms, smooth_spw=smooth_spw)

        # Should have all scratch columns in output
        cd = th.getColDesc(self.outputms, 'DATA')
        self.assertGreater(len(cd), 0, 'DATA column does not exist')
        cc = th.getColDesc(self.outputms, 'CORRECTED_DATA')
        self.assertGreater(len(cc), 0, 'CORRECTED_DATA does not exist')

        # Compare results
        self._compare_results(self.msfile, self.outputms, smooth_spw)

    def test_smooth_spw_str(self):
        '''hanningsmooth: Use string parameter smooth_spw to select only some of the SPWs to be smoothed'''

        self.outputms = 'hanning_smooth_spw.ms'
        smooth_spw = '6,7,8,9'
        smooth_spw_intvec = [6,7,8,9]
        hanningsmooth(vis=self.msfile, outputvis=self.outputms, smooth_spw=smooth_spw)

        # Should have all scratch columns in output
        cd = th.getColDesc(self.outputms, 'DATA')
        self.assertGreater(len(cd), 0, 'DATA column does not exist')
        cc = th.getColDesc(self.outputms, 'CORRECTED_DATA')
        self.assertGreater(len(cc), 0, 'CORRECTED_DATA does not exist')

        # Compare results
        self._compare_results(self.msfile, self.outputms, smooth_spw_intvec)

    def test_smooth_spw_str_spaces(self):
        '''hanningsmooth: Use string parameter smooth_spw with spaces to select only some of the SPWs to be smoothed'''

        self.outputms = 'hanning_smooth_spw.ms'
        smooth_spw = ' 6,7 ,8, 9'
        smooth_spw_intvec = [6,7,8,9]
        hanningsmooth(vis=self.msfile, outputvis=self.outputms, smooth_spw=smooth_spw)

        # Should have all scratch columns in output
        cd = th.getColDesc(self.outputms, 'DATA')
        self.assertGreater(len(cd), 0, 'DATA column does not exist')
        cc = th.getColDesc(self.outputms, 'CORRECTED_DATA')
        self.assertGreater(len(cc), 0, 'CORRECTED_DATA does not exist')

        # Compare results
        self._compare_results(self.msfile, self.outputms, smooth_spw_intvec)

    def test_smooth_spw_str_default(self):
        '''hanningsmooth: Use default empty parameter smooth_spw to select only some of the SPWs to be smoothed'''

        self.outputms = 'hanning_smooth_spw.ms'
        smooth_spw = ''
        smooth_spw_intvec = [0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15]
        hanningsmooth(vis=self.msfile, outputvis=self.outputms, smooth_spw=smooth_spw)

        # Should have all scratch columns in output
        cd = th.getColDesc(self.outputms, 'DATA')
        self.assertGreater(len(cd), 0, 'DATA column does not exist')
        cc = th.getColDesc(self.outputms, 'CORRECTED_DATA')
        self.assertGreater(len(cc), 0, 'CORRECTED_DATA does not exist')

        # Compare results
        self._compare_results(self.msfile, self.outputms, smooth_spw_intvec)

    def test_smooth_spw_str_range(self):
        '''hanningsmooth: Use string parameter smooth_spw with range syntax to select only some of the SPWs to be smoothed'''

        self.outputms = 'hanning_smooth_spw.ms'
        smooth_spw = '6~8,13'
        smooth_spw_intvec = [6,7,8,13]
        hanningsmooth(vis=self.msfile, outputvis=self.outputms, smooth_spw=smooth_spw)

        # Should have all scratch columns in output
        cd = th.getColDesc(self.outputms, 'DATA')
        self.assertGreater(len(cd), 0, 'DATA column does not exist')
        cc = th.getColDesc(self.outputms, 'CORRECTED_DATA')
        self.assertGreater(len(cc), 0, 'CORRECTED_DATA does not exist')

        # Compare results
        self._compare_results(self.msfile, self.outputms, smooth_spw_intvec)

    def test_smooth_spw_int(self):
        '''hanningsmooth: Use string parameter smooth_spw with range syntax to select only some of the SPWs to be smoothed'''

        self.outputms = 'hanning_smooth_spw.ms'
        smooth_spw = 5
        smooth_spw_intvec = [5]
        hanningsmooth(vis=self.msfile, outputvis=self.outputms, smooth_spw=smooth_spw)

        # Should have all scratch columns in output
        cd = th.getColDesc(self.outputms, 'DATA')
        self.assertGreater(len(cd), 0, 'DATA column does not exist')
        cc = th.getColDesc(self.outputms, 'CORRECTED_DATA')
        self.assertGreater(len(cc), 0, 'CORRECTED_DATA does not exist')

        # Compare results
        self._compare_results(self.msfile, self.outputms, smooth_spw_intvec)

    def test_smooth_spw_intvec_with_selection(self):
        '''hanningsmooth: Use vector integer parameter smooth_spw to select only some of the SPWs to be smoothed. Additional selection is performed'''

        self.outputms = 'hanning_smooth_spw.ms'
        smooth_spw = [6,7,8,9]
        spw_sel = '5,6,7,8,9,10'
        spw_sel_intvec = [5,6,7,8,9,10]
        hanningsmooth(vis=self.msfile, outputvis=self.outputms, spw=spw_sel, smooth_spw=smooth_spw)

        # Should have all scratch columns in output
        cd = th.getColDesc(self.outputms, 'DATA')
        self.assertGreater(len(cd), 0, 'DATA column does not exist')
        cc = th.getColDesc(self.outputms, 'CORRECTED_DATA')
        self.assertGreater(len(cc), 0, 'CORRECTED_DATA does not exist')

        # Compare results
        self._compare_results(self.msfile, self.outputms, smooth_spw, spw_sel_intvec)

    def test_smooth_spw_intvec_with_selection_no_overlap(self):
        '''hanningsmooth: Use vector integer parameter smooth_spw to select only some of the SPWs to be smoothed. Additional selection is performed which does not overlap'''

        self.outputms = 'hanning_smooth_spw.ms'
        smooth_spw = [1,2,3]
        spw_sel = '7,8,9'
        spw_sel_intvec = [7,8,9]
        hanningsmooth(vis=self.msfile, outputvis=self.outputms, spw=spw_sel, smooth_spw=smooth_spw)

        # Should have all scratch columns in output
        cd = th.getColDesc(self.outputms, 'DATA')
        self.assertGreater(len(cd), 0, 'DATA column does not exist')
        cc = th.getColDesc(self.outputms, 'CORRECTED_DATA')
        self.assertGreater(len(cc), 0, 'CORRECTED_DATA does not exist')

        # Compare results
        self._compare_results(self.msfile, self.outputms, smooth_spw, spw_sel_intvec)

    def test_smooth_spw_intvec_with_selection_gap(self):
        '''hanningsmooth: Use vector integer parameter smooth_spw to select only some of the SPWs to be smoothed. Additional selection is performed, one non-overlapping SPW'''

        self.outputms = 'hanning_smooth_spw.ms'
        smooth_spw = [7,9]
        spw_sel = '7,8,9'
        spw_sel_intvec = [7,8,9]
        hanningsmooth(vis=self.msfile, outputvis=self.outputms, spw=spw_sel, smooth_spw=smooth_spw)

        # Should have all scratch columns in output
        cd = th.getColDesc(self.outputms, 'DATA')
        self.assertGreater(len(cd), 0, 'DATA column does not exist')
        cc = th.getColDesc(self.outputms, 'CORRECTED_DATA')
        self.assertGreater(len(cc), 0, 'CORRECTED_DATA does not exist')

        # Compare results
        self._compare_results(self.msfile, self.outputms, smooth_spw, spw_sel_intvec)

    def test_smooth_spw_wrong_str_format(self):
        '''hanningsmooth: check a wrongly format parameter throws'''


        self.outputms = 'hanning_smooth_spw.ms'
        smooth_spw = '6~8,bad'
        with self.assertRaises(ValueError) :
            hanningsmooth(vis=self.msfile, outputvis=self.outputms, smooth_spw=smooth_spw)

        smooth_spw = '6,,,8'
        with self.assertRaises(ValueError) :
            hanningsmooth(vis=self.msfile, outputvis=self.outputms, smooth_spw=smooth_spw)

        smooth_spw = '6, ,8'
        with self.assertRaises(ValueError) :
            hanningsmooth(vis=self.msfile, outputvis=self.outputms, smooth_spw=smooth_spw)

        smooth_spw = [3,4.5]
        with self.assertRaises(AssertionError) :
            hanningsmooth(vis=self.msfile, outputvis=self.outputms, smooth_spw=smooth_spw)

        smooth_spw = 4.5
        with self.assertRaises(AssertionError) :
            hanningsmooth(vis=self.msfile, outputvis=self.outputms, smooth_spw=smooth_spw)

        smooth_spw = [3,'bad']
        with self.assertRaises(AssertionError) :
            hanningsmooth(vis=self.msfile, outputvis=self.outputms, smooth_spw=smooth_spw)

class Cleanup(test_base):

    def tearDown(self):
        shutil.rmtree('ngc5921_ut.ms', ignore_errors=True)
        shutil.rmtree('ALMA-data-mst-science-testing-CAS-5013-one-baseline-one-timestamp.ms', ignore_errors=True)
        
    def test_runTest(self):
        '''hanningsmooth: Cleanup'''
        pass
            
if __name__ == '__main__':
    unittest.main()
