from __future__ import absolute_import
import string

import pipeline.infrastructure as infrastructure
import pipeline.infrastructure.vdp as vdp
import pipeline.infrastructure.api as api

from . import adapters
from . import bandpassworker
from . import channelbandpass
from . import common

__all__ = [
    'PolynomialBandpass',
    'PolynomialBandpassInputs'
]


from pipeline.h.heuristics import echoheuristic as echoheuristic
from pipeline.hif.heuristics import bporder as bporder

LOG = infrastructure.get_logger(__name__)


class PolynomialBandpassInputs(common.VdpCommonBandpassInputs):

    @property
    def bandtype(self):
        return 'BPOLY'

    @vdp.VisDependentProperty
    def degamp (self):
        return bporder.BPOrder()

    @degamp.convert
    def degamp (self, value):
        return echoheuristic.EchoHeuristic(value)

    @vdp.VisDependentProperty
    def degphase (self):
        return bporder.BPOrder()

    @degphase.convert
    def degphase (self, value):
        return echoheuristic.EchoHeuristic(value)

    def __init__(self, context, output_dir=None, vis=None, caltable=None, 
        field=None, spw=None, antennas=None, intent=None,
        solint=None, combine=None, refant=None, minblperant=None,
        solnorm=None, fillgaps=None, degamp=None, degphase=None,
        visnorm=None, maskcenter=None, maskedge=None, append=None,
        opacity=None, parang=None):

        self.context = context
        self.vis = vis
        self.output_idr = outputdir
        self.field = field
        self.intent = intent
        self.spw = spw
        self.antennas = antennas
        self.solint = solint
        self.combine = combine
        self.refant = refant
        self.minblperant = minblperant
        self.solnorm = solnorm
        self.fillgaps = fillgaps
        self.degamp = degamp
        self.degphase = degphase
        self.visnorm = visnorm
        self.maskcenter = maskcenter
        self.maskedge = maskedge
        self.append = append
        self.opacity = opacity
        self.parang = parang

# Eextending BandpassWorker allows us to use its analyse method, otherwise we'd extend
# StandardTaskTemplate.

class PolynomialBandpass(bandpassworker.BandpassWorker):
    Inputs = PolynomialBandpassInputs

    def prepare(self):
        inputs = self.inputs
        results = common.BandpassResults()

        # If both degamp and degphase are EchoHeuristics, meaning the user has
        # supplied both values, then we don't need to generate a channel
        # bandpass for analysis.
        if not isinstance(inputs.degamp, echoheuristic.EchoHeuristic)\
            and not isinstance(inputs.degphase, echoheuristic.EchoHeuristic):
            pass
            # TODO: optimise this!

        # The heuristics compute the required polynomial order by 
        # analysing a channel bandpass caltable. Generate this caltable
        # and add the result to our subtasks. 
        channelbandpass_result = self._do_channelbandpass()
        results.preceding.append(channelbandpass_result)

        # Adapt the heuristics so that we can call them with pipeline domain
        # objects
        degamp_fn = adapters.DegAmpAdapter(inputs.degamp)
        degphase_fn = adapters.DegPhaseAdapter(inputs.degphase)
        
        # fix the caltable name to its current value so that all appends go
        # into the one file
        inputs.caltable = inputs.caltable
    
        # Loop through the caltables generated by the channel bandpass,
        # computing the best parameters as we go.
        append = False
        tasks = []
        for caltable in channelbandpass_result.final:            
            worker_inputs = self._get_bandpassworker_inputs()
            
            for spw in string.split(caltable.spw, ','):
                worker_inputs.append = append
                worker_inputs.spw = spw
                worker_inputs.degamp = degamp_fn(inputs.ms, caltable, spw)
                worker_inputs.degphase = degphase_fn(inputs.ms, caltable, spw)
    
                task = bandpassworker.BandpassWorker(worker_inputs)
                tasks.append(task)
    
                append = True

        for task in tasks:
            worker_result = self._executor.execute(task)
            results.pool.extend(worker_result.final)
         
        return results

    def _get_bandpassworker_inputs(self):
        # we basically want to create a new worker inputs and then copy across
        # all values from our polynomial inputs to the worker inputs.
        poly_inputs = self.inputs
        poly_task_args = poly_inputs._get_task_args()        
        return bandpassworker.BandpassWorker.Inputs(poly_inputs.context,
            bandtype=poly_inputs.bandtype, **poly_task_args)

    def _do_channelbandpass(self):
        inputs = self.inputs
        bandpass_inputs = channelbandpass.ChannelBandpass.Inputs(inputs.context,
            vis         = inputs.vis, 
            caltable    = None,
            # data selection arguments
            field       = inputs.field,
            spw         = inputs.spw,
            intent      = inputs.intent,
            # solution parameters
            solint      = inputs.solint,
            combine     = inputs.combine,
            refant      = inputs.refant,
            minblperant = inputs.minblperant,
            minsnr      = None,
            solnorm     = inputs.solnorm,
            fillgaps    = inputs.fillgaps)

        bandpass_task = channelbandpass.ChannelBandpass(bandpass_inputs)
        return self._executor.execute(bandpass_task, merge=False)
